<?php

namespace App\Http\Controllers\API\V1\Inventory;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\ItemType;
use Illuminate\Http\Request;
use App\Classes\Inventory\ItemTypeHelper;
use App\Models\Main\ScreenSub;
use Illuminate\Validation\Rule;
class ItemTypeController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-003','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'rows'   => $request['rows'] ?? null ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-003', $lang)->sub_title ?? '';
        $this->return['items']       = ItemType::all_item_types($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-003','add');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-003' , $lang)->sub_title ?? '';

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-003','add');

        $insert                    = $request->only(['name_en', 'name_ar']);

        $insert['add_user']        = auth()->user()->id;


        $validator = validator()->make($insert, ItemType::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $this->return['id'] = ItemType::add_item_type($insert);

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-003','view');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-003', $lang)->sub_title ?? '';
        $this->return['data']          = ItemType::get_item_type($id, $lang);

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-003','edit');

        if(!(new ItemTypeHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.update_fail"));
        }

        $update                    = $request->only(['name_en', 'name_ar']);
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('item_types')->ignore($id,'item_type_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('item_types')->ignore($id,'item_type_id'),
            ]
        ];

        $validator  = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        ItemType::update_item_type($update, $id);

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-003','delete');

        if(!(new ItemTypeHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        ItemType::delete_item_type($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = ItemType::find($id);

        if($item->item_type_active){
            $this->checkPermission('03-003','delete');

            if(!(new ItemTypeHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            ItemType::delete_item_type($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-003','edit');

            if(!(new ItemTypeHelper)->can_edit_or_delete($id)){
                return $this->sendError( trans('error.update_fails'), []);
            }

            ItemType::update_item_type(array('item_type_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
