<?php

namespace App\Http\Controllers\API\V1\Inventory;
use App\Classes\Inventory\StockTransactionHelper;
use App\Classes\ItemStock;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\Item;
use App\Models\Inventory\StockTransaction;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\ScreenSub;
use Illuminate\Http\Request;
use DB;

class StockTransactionController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        if(!@in_array('1', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError( 'view permission needed #03-007', $this->data, 200);
        }

        $this->filter = array(
            'lang'          => $lang ,
            'active'        => $request['active'] ?? 1 ,
            'rows'          => $request['rows'] ?? 20 ,
            'page'          => $request['page'] ?? 1 ,
            'word'          => $request['word'] ?? null ,
            'branch'        => $this->data['current_branch'] ,
            'finance_year'  => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-007', $lang)->sub_title ?? '';
        $this->return['items']       = StockTransaction::all_stock_transactions($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        if(!@in_array('2', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError('create permission needed #03-007', [], 200);
        }

        $this->return['screen_name']              = ScreenSub::get_screen_sub_by_screen_code('03-007' , $lang)->sub_title ?? '';
        $this->return['stock_transactions_types'] = DB::table('stock_transaction_types')->select('stock_transaction_type_id', 'stock_transaction_type_title_'.$lang. ' as title', 'account_tree_id')->get();
        $this->return['warehouses']               = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']                    = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']        = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        if(!@in_array('2', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError('create permission needed #03-007', [], 200);
        }

        $insert = $request->only(['stock_transaction_type_id', 'stock_transaction_date', 'stock_transaction_description', 'warehouse_id', 'items']);

        $year   = array_search(date("Y", strtotime($insert['stock_transaction_date'])), $this->data['years']);

        if($year == ''){
            return $this->sendError( trans("error.finance_year_not_aval"));
        }

        $codes = (new StockTransactionHelper)->get_next_codes($insert['stock_transaction_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                     => $this->data['current_branch'],
            'finance_year_id'               => $year,
            'year_ser'                      => $codes['year_ser'],
            'month_ser'                     => $codes['month_ser'],
            'branch_ser'                    => $codes['branch_ser'],
            'daily_journal_m_id'            => 0,
            'stock_transaction_type_id'     => $insert['stock_transaction_type_id'],
            'stock_transaction_date'        => $insert['stock_transaction_date'],
            'stock_transaction_description' => $insert['stock_transaction_description'],
            'warehouse_id'                  => $insert['warehouse_id'],
            'add_user'                      => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, StockTransaction::$master_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $m_id = StockTransaction::add_stock_transaction_m($master_seed);

        if (isset($insert['items']) && is_array($insert['items'])){
            foreach ($insert['items'] as $item){
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }
                }
                $details_seed = array(
                    'stock_transaction_m_id' => $m_id,
                    'item_id'                => $item['item_id'],
                    'unit_id'                => $item['unit_id'],
                    'unit_type'              => $item['unit_type'] ?? 0,
                    'quantity'               => $item['quantity'] ?? 0,
                    'price'                  => $item['price'] ?? 0,
                    'total_price'            => ($item['price'] * $item['quantity']),
                    'add_user'               => auth()->user()->id
                );

                $validator = validator()->make($details_seed, StockTransaction::$details_rules);

                if($validator->fails())
                {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = StockTransaction::add_stock_transaction_d($details_seed);
                $warehouse_id = $insert['warehouse_id'];
                $qty = $item['quantity'];
                $source_map = [
                    1 => ItemStock::Incoming_source, // + stock
                    2 => ItemStock::Outcoming_source, // - stock
                    3 => ItemStock::Stock_Difference_source, // + stock
                    4 => ItemStock::Gift_Samples_source, // + stock
                    5 => ItemStock::Damage_source, // - stock
                ];
                $source = $source_map[$insert['stock_transaction_type_id']];
                // update item
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $warehouse_id, $source, $m_id, $d_id, $qty, 0, $item['unit_id'], $item['unit_type'], 0);
            }
        }

        $updateMasterValues   = (new StockTransactionHelper)->update_master_values($m_id);
        $generateDailyJournal = (new StockTransactionHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse([], trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        if(!@in_array('1', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError('show permission needed #03-007', [], 200);
        }

        $this->return['screen_name']              = ScreenSub::get_screen_sub_by_screen_code('03-007' , $lang)->sub_title ?? '';
        $this->return['master']                   = StockTransaction::get_stock_transaction_m($id, $lang);
        $this->return['details']                  = StockTransaction::get_stock_transaction_d($id, $lang);

        $this->return['stock_transactions_types'] = DB::table('stock_transaction_types')->select('stock_transaction_type_id', 'stock_transaction_type_title_'.$lang. ' as title', 'account_tree_id')->get();
        $this->return['warehouses']               = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']                    = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']        = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        if(!@in_array('3', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError( 'update permission needed #03-007', [], 200);
        }

        $update = $request->only(['stock_transaction_type_id', 'stock_transaction_date', 'stock_transaction_description', 'warehouse_id']);

        StockTransaction::update_stock_transaction_m($update, $id);
        return $this->sendResponse($this->return, trans('main.update_success'));
    }

//    public function update($lang, $id, Request $request)
//    {
//        if(!@in_array('3', $this->data['mypermissions']['03-007'][0]->permission)){
//            return $this->sendError('update permission needed #03-007', [], 200);
//        }
//
//        $update = $request->only(['stock_transaction_type_id', 'stock_transaction_date', 'stock_transaction_description', 'warehouse_id', 'items']);
//
//        if(!isset($update['items']) || !is_array($update['items'])){
//            return $this->sendError(trans("error.items"));
//        }
//
//        $master = StockTransaction::get_stock_transaction_m($id);
//
//        if(!isset($master->stock_transaction_m_id) || $master->branch_id != $this->data['current_branch'] || !(new StockTransactionHelper)->can_edit_or_delete($id)){
//            return $this->sendError(trans("error.update_fail"));
//        }
//
//        $update_seed = collect($update)->only('stock_transaction_type_id', 'stock_transaction_date', 'stock_transaction_description', 'warehouse_id')->all();
//        $rules       = collect(StockTransaction::$master_rules)->only('stock_transaction_type_id', 'stock_transaction_date', 'stock_transaction_description', 'warehouse_id')->all();
//
//        $validator = validator()->make($update_seed, $rules);
//
//        if($validator->fails())
//        {
//            return $this->sendError(trans('error.update_fails'), $validator->errors());
//        }
//
//        DB::beginTransaction();
//
//        StockTransaction::update_stock_transaction_m($update_seed, $id);
//        StockTransaction::delete_stock_transaction_d($id);
//
//        if (isset($update['items']) && is_array($update['items'])) {
//            foreach ($update['items'] as $item){
//                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
//                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
//                    if ($item['unit_id'] == $selectedItem->big_unit_id){
//                        $item['unit_type'] = 3;
//                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
//                        $item['unit_type'] = 2;
//                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
//                        $item['unit_type'] = 1;
//                    }
//                }
//                $details_seed = array(
//                    'stock_transaction_m_id' => $id,
//                    'item_id'                => $item['item_id'],
//                    'unit_id'                => $item['unit_id'],
//                    'unit_type'              => $item['unit_type'] ?? 0,
//                    'quantity'               => $item['quantity'] ?? 0,
//                    'price'                  => $item['price'] ?? 0,
//                    'total_price'            => ($item['price'] * $item['quantity']),
//                    'add_user'               => auth()->user()->id
//                );
//
//                $validator = validator()->make($details_seed, StockTransaction::$details_rules);
//
//                if($validator->fails())
//                {
//                    return $this->sendError(trans('error.update_fails'), $validator->errors());
//                }
//
//                StockTransaction::add_stock_transaction_d($details_seed);
//            }
//        }
//
//        $updateMasterValues   = (new StockTransactionHelper)->update_master_values($id);
//        $generateDailyJournal = (new StockTransactionHelper)->generate_daily_journal($id);
//
//        DB::commit();
//        return $this->sendResponse($this->return, trans('main.update_success'));
//    }

    public function destroy($lang, $id)
    {
        if(!@in_array('4', $this->data['mypermissions']['03-007'][0]->permission)){
            return $this->sendError('delete permission needed #03-007', [], 200);
        }

        $master = StockTransaction::get_stock_transaction_m($id);

        if(!isset($master->stock_transaction_m_id) || $master->branch_id != $this->data['current_branch'] || !(new StockTransactionHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        StockTransaction::delete_stock_transaction_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = StockTransaction::find($id);

        if($item->stock_transaction_active){
            if(!@in_array('4', $this->data['mypermissions']['03-007'][0]->permission)){
                return $this->sendError('update permission needed #03-007', [], 200);
            }

            if(!(new StockTransactionHelper)->can_edit_or_delete($id)){
                return $this->sendError( trans('error.delete_fails'), []);
            }

            StockTransaction::delete_stock_transaction_m($id);
            $this->return['active'] = false;
        }else{
            if(!@in_array('3', $this->data['mypermissions']['03-007'][0]->permission)){
                return $this->sendError('update permission needed #03-007', [], 200);
            }

            if(!(new StockTransactionHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            StockTransaction::update_stock_transaction_m(array('stock_transaction_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
