<?php

namespace App\Http\Controllers\API\V1\Inventory;

use App\Classes\Inventory\StocktakingHelper;
use App\Classes\ItemStock;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\Item;
use App\Models\Inventory\Stocktaking;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\ScreenSub;
use Illuminate\Http\Request;
use DB;

class StocktakingController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        if (!@in_array('1', $this->data['mypermissions']['03-009'][0]->permission)) {
            return $this->sendError('view permission needed #03-009', $this->data, 200);
        }

        $this->filter = array(
            'lang'         => $lang,
            'active'       => $request['active'] ?? 1,
            'rows'         => $request['rows'] ?? 20,
            'page'         => $request['page'] ?? 1,
            'word'         => $request['word'] ?? null,
            'warehouse_id' => $request['warehouse_id'] ?? null,
            'from_date'    => $request['from_date'] ?? null,
            'to_date'      => $request['to_date'] ?? null,
            'branch'       => $this->data['current_branch'],
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-009', $lang)->sub_title ?? '';
        $this->return['items']       = Stocktaking::all_stocktakings($this->filter);
        $this->return['warehouses']  = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        if (!@in_array('2', $this->data['mypermissions']['03-009'][0]->permission)) {
            return $this->sendError('create permission needed #03-009', [], 200);
        }

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-009', $lang)->sub_title ?? '';
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        if(!@in_array('2', $this->data['mypermissions']['03-009'][0]->permission)){
            return $this->sendError( 'create permission needed #03-009', [], 200);
        }

        $insert = $request->only(['stocktaking_date', 'warehouse_id', 'items']);
        $year   = array_search(date("Y", strtotime($insert['stocktaking_date'])), $this->data['years']);

        if ($year == '') {
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        $codes = (new StocktakingHelper)->get_next_codes($insert['stocktaking_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'          => $this->data['current_branch'],
            'finance_year_id'    => $year,
            'year_ser'           => $codes['year_ser'],
            'month_ser'          => $codes['month_ser'],
            'branch_ser'         => $codes['branch_ser'],
            'daily_journal_m_id' => 0,
            'stocktaking_date'   => $insert['stocktaking_date'],
            'warehouse_id'       => $insert['warehouse_id'],
            'add_user'           => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, Stocktaking::$master_rules);

        if ($validator->fails()) {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $m_id = Stocktaking::add_stocktaking_m($master_seed);

        if (isset($insert['items']) && is_array($insert['items'])) {
            foreach ($insert['items'] as $item) {
                if (isset($item['unit_id']) && !is_null($item['unit_id'])) {
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id) {
                        $item['unit_type'] = 3;
                    } elseif ($item['unit_id'] == $selectedItem->mid_unit_id) {
                        $item['unit_type'] = 2;
                    } elseif ($item['unit_id'] == $selectedItem->small_unit_id) {
                        $item['unit_type'] = 1;
                    }
                }

                $item_check = (new ItemStock)->get_item_stock(array('item' => $item['item_id'], 'warehouse' => $insert['warehouse_id']));

                if ($item_check->isEmpty()) {
                    $stock = 0;
                } elseif ($item_check[0]->default_report_unit_type == 1) {
                    $stock = $item_check[0]->stock_in_small;
                } elseif ($item_check[0]->default_report_unit_type == 2) {
                    $stock = $item_check[0]->stock_in_mid;
                } elseif ($item_check[0]->default_report_unit_type == 3) {
                    $stock = $item_check[0]->stock_in_big;
                }
                
                $diffrance = (($item['stocktaking_balance'] ?? 0) - ($stock));
                $qty = $item['stocktaking_balance'] ?? 0;

                $details_seed = array(
                    'stocktaking_m_id'    => $m_id,
                    'item_id'             => $item['item_id'],
                    'unit_id'             => $item['unit_id'],
                    'unit_type'           => $item['unit_type'],
                    'current_balance'     => $stock,
                    'stocktaking_balance' => $qty,
                    'difference'          => $diffrance,
                    'add_user'            => auth()->user()->id
                );

                $validator = validator()->make($details_seed, Stocktaking::$details_rules);

                if ($validator->fails()) {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = Stocktaking::add_stocktaking_d($details_seed);
                // update Item Stock
                $source = ItemStock::Stocktaking_source;
                
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $insert['warehouse_id'], $source, $m_id, $d_id, $qty, $diffrance, $item['unit_id'], $item['unit_type'], 0);
            }
        }

        $updateMasterValues   = (new StocktakingHelper)->update_master_values($m_id);
        $generateDailyJournal = (new StocktakingHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse([], trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        if (!@in_array('3', $this->data['mypermissions']['03-009'][0]->permission)) {
            return $this->sendError('update permission needed #03-009', [], 200);
        }

        $update = $request->only(['stocktaking_date', 'warehouse_id']);

        Stocktaking::update_stocktaking_m($update, $id);
        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function show($lang, $id)
    {
        if (!@in_array('1', $this->data['mypermissions']['03-009'][0]->permission)) {
            return $this->sendError('show permission needed #03-009', [], 200);
        }

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-009', $lang)->sub_title ?? '';
        $this->return['master']            = Stocktaking::get_stocktaking_m($id, $lang);
        $this->return['details']           = Stocktaking::get_stocktaking_d($id, $lang);

        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function destroy($lang, $id)
    {
        if (!@in_array('4', $this->data['mypermissions']['03-005'][0]->permission)) {
            return $this->sendError('delete permission needed #03-005', [], 200);
        }

        $master = Stocktaking::get_stocktaking_m($id);

        if (!isset($master->stocktaking_m_id) || $master->branch_id != $this->data['current_branch'] || !(new StocktakingHelper)->can_edit_or_delete($id)) {
            return $this->sendError(trans("error.delete_fail"));
        }

        Stocktaking::delete_stocktaking_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = Stocktaking::find($id);
        if ($item->stocktaking_active) {
            if (!@in_array('4', $this->data['mypermissions']['03-005'][0]->permission)) {
                return $this->sendError('update permission needed #03-005', [], 200);
            }

            if (!(new StocktakingHelper)->can_edit_or_delete($id)) {
                return $this->sendError(trans('error.delete_fails'), []);
            }

            Stocktaking::delete_stocktaking_m($id);
            $this->return['active'] = false;
        } else {

            if (!@in_array('3', $this->data['mypermissions']['03-005'][0]->permission)) {
                return $this->sendError('update permission needed #03-005', [], 200);
            }

            if (!(new StocktakingHelper)->can_edit_or_delete($id)) {
                return $this->sendError(trans('error.update_fails'), []);
            }

            Stocktaking::update_stocktaking_m(array('stocktaking_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function get_stock_items($lang, Request $request)
    {
        if (!@in_array('1', $this->data['mypermissions']['03-009'][0]->permission)) {
            return $this->sendError('view permission needed #03-009', $this->data, 200);
        }

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-009', $lang)->sub_title ?? '';
        $this->return['items']       = (new ItemStock)->get_item_stock(array('branch' => $request['branch'], 'warehouse' => $request['warehouse'], 'lang' => $lang));

        return $this->sendResponse($this->return);
    }
}
