<?php

namespace App\Http\Controllers\API\V1\Sales;
use App\Classes\Excel\InvoicesExport;
use App\Classes\General\Upload;
use App\Classes\ItemStock;
use App\Classes\Sales\DeliveryNoteHelper;
use App\Classes\Sales\ProformaInvoiceHelper;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Sales\Customer;
use App\Models\Sales\DeliveryNote;
use App\Models\Sales\ProformaInvoice;
use App\Models\Sales\SalesAgent;
use Illuminate\Http\Request;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Models\Finance\Currency;
use PDF;
use Excel;

class DeliveryNoteController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('06-007' , 'view');

        $this->filter = array(
            'lang'                => $lang ,
            'active'              => $request['active'] ?? 1 ,
            'status'              => $request['status'] ?? 1 ,
            'rows'                => $request['rows'] ?? 20 ,
            'page'                => $request['page'] ?? 1 ,
            'word'                => $request['word'] ?? null ,
            'customer_id'         => $request['customer_id'] ?? null ,
            'proforma_invoice_id' => $request['proforma_invoice_id'] ?? null ,
            'branch'              => $this->data['current_branch'] ,
            'finance_year'        => $this->data['current_year']
        );

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-007', $lang)->sub_title ?? '';
        $this->return['items']             = DeliveryNote::all_delivery_notes($this->filter);
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang));
        // $this->return['proforma_invoices'] = ProformaInvoice::all_proforma_invoices(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('06-007' , 'add');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-007', $lang)->sub_title ?? '';
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang));
        // $this->return['proforma_invoices'] = ProformaInvoice::all_proforma_invoices(array('active' => 1, 'lang' => $lang, 'is_closed' => 0));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('06-007' , 'add');

        $insert = $request->only(['customer_id', 'proforma_invoice_id', 'driver_id', 'car_number', 'delivery_note_date',
                                  'delivery_note_time', 'delivery_note_description', 'sales_agent_id' , 'items']);

        $year   = array_search(date("Y", strtotime($insert['delivery_note_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        if(!isset($insert['items']) || !is_array($insert['items']) || count($insert['items']) < 1 ){
            return $this->sendError('Erro In Items');
        }

        if(isset($insert['proforma_invoice_id']) && $insert['proforma_invoice_id'] > 0 ){
            $check = DB::table('proforma_invoice_m')
                ->where('proforma_invoice_m_id' , $insert['proforma_invoice_id'] )
                ->where('proforma_invoice_status' , 1 )
                ->first();
            if(!isset($check->proforma_invoice_m_id)){
                return $this->sendError('Erro In Proforma Invoice ID');
            }
        }

        $codes = (new DeliveryNoteHelper)->get_next_codes($insert['delivery_note_date'], $year, $this->data['current_branch']);
        
        $insert['branch_id']         = $this->data['current_branch'];
        $insert['finance_year_id']   = $year;
        $insert['month_ser']         = $codes['year_ser'];
        $insert['month_ser']         = $codes['month_ser'];
        $insert['branch_ser']        = $codes['branch_ser'];
        $insert['add_user']          = auth()->user()->id;
        unset($insert['items']);

        $validator = validator()->make($insert, DeliveryNote::$master_rules);
        if($validator->fails()){
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }



        DB::beginTransaction();
            if($insert['proforma_invoice_id'] > 0 ){
                (new ProformaInvoiceHelper)->update_master_values($insert['proforma_invoice_id']);
            }

            $m_id   = DeliveryNote::add_delivery_note_m($insert);
            $items  = (new DeliveryNoteHelper)->prepare_details( $m_id , $insert['proforma_invoice_id'] , $request['items'] , 0 );

            if(count($items) < 1 ){
                DB::rollback();
                return $this->sendError(trans('error.add_fails'), []);
            }

            $source = ItemStock::Delivery_Note_source; // - stock
            foreach ($items as $item){
                $validator = validator()->make($item, ProformaInvoice::$details_rules);
                if($validator->fails())
                    return $this->sendError(trans('error.add_fails'), $validator->errors());

                if($item['rem_qty'] < 0){
                    return $this->sendError(trans('error.add_fails'), [$item['rem_qty'] , $item['quantity']] );
                }
                $d_id = DeliveryNote::add_delivery_note_d($item);
                $qty = $item['quantity'];
                // Item Stock
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $m_id, $d_id, $qty, 0 , $item['unit_id'], $item['unit_type'], 0);
            }

            (new DeliveryNoteHelper)->update_master_values($m_id);
            (new DeliveryNoteHelper)->generate_daily_journal($m_id);

            if($insert['proforma_invoice_id'] > 0 ){
                (new ProformaInvoiceHelper)->update_master_values($insert['proforma_invoice_id']);
            }

        DB::commit();
        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('06-007' , 'view');
        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-007', $lang)->sub_title ?? '';
        $this->return['master']            = DeliveryNote::get_delivery_note_m($id, $lang);
        $this->return['details']           = DeliveryNote::get_delivery_note_d($id, $lang);
        $this->return['files']             = DeliveryNote::get_delivery_note_files($id);

        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang));
        $this->return['proforma_invoices'] = ProformaInvoice::all_proforma_invoices(array('active' => 1, 'lang' => $lang));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('06-007' , 'edit');

        // return $this->sendError(trans("error.update_fail"));

        $update = $request->only(['customer_id', 'proforma_invoice_id', 'driver_id', 'car_number', 'delivery_note_date', 'delivery_note_time', 'delivery_note_description', 'sales_agent_id' , 'items']);


        $year   = array_search(date("Y", strtotime($update['delivery_note_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        if(!isset($update['items']) || !is_array($update['items']) || count($update['items']) < 1 ){
            return $this->sendError('Erro In Items');
        }

        foreach ($update['items'] as $key => $value) {
            // $update['items'][$key]['quantity'] = doubleval( $update['items'][$key]['delivered_quantity'] );
        }

        DB::beginTransaction();


            // DB::table('delivery_note_d')
            //     ->where('delivery_note_m_id' , $id )
            //     ->delete();

            // $items  = (new DeliveryNoteHelper)->prepare_details( $id , $update['proforma_invoice_id'] , $update['items'] , 1 , 0 );

            // if(count($items) < 1 ){
            //     DB::rollback();
            //     return $this->sendError(trans('error.add_fails'), []);
            // }

            unset($update['items']);
            DeliveryNote::update_delivery_note_m($update, $id);

            // foreach ($items as $item){

            //     $validator = validator()->make($item, ProformaInvoice::$details_rules);
            //     if($validator->fails())
            //         return $this->sendError(trans('error.add_fails'), $validator->errors());

            //     if (isset($item['proforma_invoice_d_id']) && $item['proforma_invoice_d_id'] > 0){

            //         $proforma = DB::table('proforma_invoice_d')
            //             ->join('proforma_invoice_m' , 'proforma_invoice_d.proforma_invoice_m_id' , '=' , 'proforma_invoice_m.proforma_invoice_m_id')
            //             ->where('proforma_invoice_d.proforma_invoice_m_id' , $update['proforma_invoice_id'])
            //             ->where('proforma_invoice_d.proforma_invoice_d_id' , $item['proforma_invoice_d_id'])
            //             ->where('proforma_invoice_active' , 1 )
            //             ->value('remaining_quantity') ?? 0  ;

            //         if($proforma < $item['quantity']){
            //             return $this->sendError(trans('error.add_fails'), [$proforma , $item['quantity']] );
            //         }

            //     }
            //     $d_id = DeliveryNote::add_delivery_note_d($item);

            // }


            (new DeliveryNoteHelper)->update_master_values($id);
            (new DeliveryNoteHelper)->generate_daily_journal($id);

            if($update['proforma_invoice_id'] > 0 ){
                (new ProformaInvoiceHelper)->update_master_values($update['proforma_invoice_id']);
            }


        DB::commit();

        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('06-007' , 'delete');

        $master = DeliveryNote::get_delivery_note_m($id);

        if(!(new DeliveryNoteHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }
        DB::beginTransaction();

            DeliveryNote::delete_delivery_note_m($id);
            (new ProformaInvoiceHelper)->update_master_values($master->proforma_invoice_id);

        DB::commit();

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $master = DeliveryNote::find($id);

        if($master->delivery_note_active){
            $this->checkPermission('06-007' , 'delete');

            if(!(new DeliveryNoteHelper)->can_edit_or_delete($id)){
                return $this->sendError( trans('error.delete_fails'), []);
            }


            DB::beginTransaction();

                DeliveryNote::delete_delivery_note_m($id);
                if($master->proforma_invoice_id > 0 ){
                    (new ProformaInvoiceHelper)->update_master_values($master->proforma_invoice_id);
                }

            DB::commit();

            $this->return['active'] = false;
        }else{
            $this->checkPermission('06-007' , 'edit');

            DB::beginTransaction();

                DeliveryNote::update_delivery_note_m(array('delivery_note_active' => 1), $id);
                if($master->proforma_invoice_id > 0 ){
                    (new ProformaInvoiceHelper)->update_master_values($master->proforma_invoice_id);
                }

            DB::commit();

            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function uploadFiles($lang, Request $request)
    {
        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'delivery_note_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('delivery_note_files')->insert([
                        'delivery_note_m_id' => $request['id'],
                        'delivery_note_d_id' => 0,
                        'file_name'          => $upload['filename'],
                        'file_path'          => $upload['url'],
                        'file_extension'     => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'delivery_note_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('delivery_note_files')->insert([
                'delivery_note_m_id' => $request['id'],
                'delivery_note_d_id' => 0,
                'file_name'          => $upload['filename'],
                'file_path'          => $upload['url'],
                'file_extension'     => $upload['extension'],
            ]);
        }

        return $this->sendResponse([], trans('main.Upload Successfully'));
    }

    public function accept_delivery_note($lang, $id, Request $request)
    {
        $quotation = DB::table('delivery_note_m')
            ->where('delivery_note_m_id', $id)
            ->update([
                'delivery_note_status' => 2
            ]);
        return $this->sendResponse($this->return, trans('main.Delivery Note Accepted Successfully'));
    }

    public function update_delivery_note_status($lang, $id, Request $request)
    {
        $quotation = DB::table('delivery_note_m')
            ->where('delivery_note_m_id', $id)
            ->update([
                'delivery_note_status' => $request['status']
            ]);

        return $this->sendResponse($this->return, trans('main.update_success'));
    }


    public function get_customer_proforma($lang, Request $request)
    {
        $this->return['items'] = DB::table('proforma_invoice_m')
            ->where('customer_id', $request['customer_id'])
            ->where('proforma_invoice_status', 1)
            ->where('proforma_invoice_active' , 1)
            ->where('sales_invoice_id' ,  0 )
            ->selectRaw('proforma_invoice_m_id as id , purchase_order_id , proforma_invoice_date , sales_agent_id')->get();
        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function get_proforma_data($lang, Request $request)
    {
        $this->return['items']  = ProformaInvoice::get_proforma_invoice_d($request['proforma_invoice_id'], $lang);
        $this->return['master'] = ProformaInvoice::get_proforma_invoice_m($request['proforma_invoice_id'], $lang);

        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function print_invoice($id, Request $request)
    {
        $input = $request->all();

        App::setLocale('ar');

        $this->data['lang']        = 'ar';
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-008' , $this->data['lang']);
        $this->data['master']      = DeliveryNote::get_delivery_note_m($id, $this->data['lang']);
        $this->data['details']     = DeliveryNote::get_delivery_note_d($id, $this->data['lang']);
        $this->data['customer']    = Customer::get_customer($this->data['master']->customer_id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);

        $temp_name                 = Setting::get_main_setting_by_code('delivery_note_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Sales.delivery_note.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }
}
