<?php

namespace App\Http\Controllers\API\V1\Finance;

use App\Classes\Finance\ExpenseCodeHelper;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Finance\AccountTree;
use App\Models\Finance\CostCenter;
use App\Models\Finance\ExpenseCode;
use Illuminate\Http\Request;
use App\Models\Main\ScreenSub;
use Illuminate\Validation\Rule;

class ExpenseCodeController  extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('02-014','view');

        $this->filter = array(
            'lang'         => $lang ,
            'active'       => $request['active'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'branch'       => $this->data['current_branch'],
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('02-014', $lang)->sub_title ?? '';
        $this->return['items']       = ExpenseCode::all_expense_codes($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('02-014','add');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('02-014', $lang)->sub_title ?? '';
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['cost_centers']  = CostCenter::active_cost_centers($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('02-014','add');

        $insert = $request->only(['name_en', 'name_ar', 'account_tree_id', 'cost_center_id']);

        $seed = [
            'name_en'         => $insert['name_en'],
            'name_ar'         => $insert['name_ar'],
            'account_tree_id' => $insert['account_tree_id'],
            'cost_center_id'  => $insert['cost_center_id'] ?? 0,
            'add_user'        => auth()->user()->id
        ];

        $validator = validator()->make($seed, ExpenseCode::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $this->return['id'] = ExpenseCode::add_expense_code($seed);
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('02-014','view');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('02-014', $lang)->sub_title ?? '';
        $this->return['item']          = ExpenseCode::get_expense_code($id, $lang);
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['cost_centers']  = CostCenter::active_cost_centers($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('02-014','edit');

        $update    = $request->only(['name_en', 'name_ar', 'account_tree_id', 'cost_center_id']);
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('expense_codes')->ignore($id,'expense_code_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('expense_codes')->ignore($id,'expense_code_id'),
            ]
        ];
        $validator = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        ExpenseCode::update_expense_code($update, $id);
        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('02-014','delete');

        ExpenseCode::delete_expense_code($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang , $id)
    {
        $item = ExpenseCode::find($id);

        if($item->expense_code_active){
            $this->checkPermission('02-014','delete');

            if(!(new ExpenseCodeHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            ExpenseCode::delete_expense_code($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('02-014','edit');

            if(!(new ExpenseCodeHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            ExpenseCode::update_expense_code(array('expense_code_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
