<?php
namespace App\Http\Controllers\API\V1\Inventory;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;

use App\Classes\Inventory\ItemHelper;
use App\Classes\General\Upload;

use App\Models\Finance\Tax;
use App\Models\Inventory\Category;
use App\Models\Inventory\Item;
use App\Models\Main\ScreenSub;
use App\Models\Finance\AccountTree;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\ItemType;
use App\Models\Main\Setting;
use Illuminate\Validation\Rule;

class ItemController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-004','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'rows'   => $request['rows'] ?? null ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-004', $lang)->sub_title ?? '';
        $this->return['items']       = Item::all_items($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-004','add');


        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-004', $lang)->sub_title ?? '';
        $this->return['account_trees']     = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['categorys']         = Category::all_categorys(array('active' => 1, 'lang' => $lang , 'category_type' => 0));
        $this->return['item_types']        = ItemType::all_item_types(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );

        $this->return['defaults']          = array(
            'purchases_revenue_id'  => intval( Setting::get_main_setting_by_code('item_default_purchases_revenue')->value ?? 0 ) ,
            'purchases_return_id'   => intval( Setting::get_main_setting_by_code('item_default_purchases_return')->value ?? 0 ),
            'purchases_discount_id' => intval( Setting::get_main_setting_by_code('item_default_purchases_discount')->value ?? 0) ,
            'sales_revenue_id'      => intval( Setting::get_main_setting_by_code('item_default_sales_revenue')->value ?? 0 ),
            'sales_return_id'       => intval( Setting::get_main_setting_by_code('item_default_sales_return')->value ?? 0 ),
            'sales_discount_id'         => intval( Setting::get_main_setting_by_code('item_default_sales_discount')->value ?? 0 ),
            'tax_id'                    => intval( Setting::get_main_setting_by_code('item_default_vat')->value ?? 0 ),
            'big_unit_id'               => intval( Setting::get_main_setting_by_code('item_default_big_unit_id')->value ?? 0 ),
            'big_unit_coefficient'      => intval( Setting::get_main_setting_by_code('item_default_big_unit_coefficient')->value ?? 0 ),
            'mid_unit_id'               => intval( Setting::get_main_setting_by_code('item_default_mid_unit_id')->value ?? 0 ),
            'mid_unit_coefficient'      => intval( Setting::get_main_setting_by_code('item_default_mid_unit_coefficient')->value ?? 0 ),
            'small_unit_id'           => intval( Setting::get_main_setting_by_code('item_default_small_unit_id')->value ?? 0 ),
            'small_unit_coefficient'  => intval( Setting::get_main_setting_by_code('item_default_small_unit_coefficient')->value ?? 0 ),
            'default_report_unit_type' => intval( Setting::get_main_setting_by_code('item_default_report_unit_type')->value ?? 0 ),

        );

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-004','add');


        $insert = $request->only(['name_en', 'name_ar', 'account_tree_id','location','item_number', 'category_id', 'item_type_id', 'big_unit_id',
                                  'big_unit_coefficient', 'mid_unit_id', 'mid_unit_coefficient', 'small_unit_id',
                                  'small_unit_coefficient', 'default_report_unit_type', 'sale_price', 'min_sale_price',
                                  'purchases_revenue_id', 'purchases_return_id', 'purchases_discount_id', 'sales_revenue_id',
                                  'sales_return_id', 'is_service', 'sales_discount_id', 'tax_id', 'time_blocks' , 'price_include_vat' , 'item_width_height' , 'pos_item' ,'description']);

        $validator = validator()->make($insert, Item::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        if(isset($request['image']) && strlen($request['image']) > 100){
            $upload = new Upload;
            $check  = $upload->uploadBase64Image($request['image'], 'items_images');
            if(!$check['check'])
                return $this->sendError( $check['msg'] , [] );
            else
                $insert['image'] = $check['url'];
        }

        $insert['add_user'] = auth()->user()->id;
        $this->return['id'] = Item::add_item($insert);
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-004','view');


        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-004', $lang)->sub_title ?? '';
        $this->return['data']              = Item::get_item($id, $lang);
        $this->return['account_trees']     = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['categorys']         = Category::all_categorys(array('active' => 1, 'lang' => $lang , 'category_type' => 0));
        $this->return['item_types']        = ItemType::all_item_types(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );

        return $this->sendResponse($this->return);
    }

    public function get_items_by_category($lang, Request $request)
    {
//        if(!@in_array('1', $this->data['mypermissions']['03-004'][0]->permission)){
//            return $this->sendError('show permission needed #03-004', [], 200);
//        }

        $this->filter = array(
            'lang'        => $lang ,
            'active'      => $request['active'] ?? 1 ,
            'rows'        => $request['rows'] ?? null ,
            'page'        => $request['page'] ?? 1 ,
            'category_id' => $request['category_id'] ?? null ,
            'word'        => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-004', $lang)->sub_title ?? '';
        $this->return['data']        = Item::all_items($this->filter);

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-004','edit');


        if(!(new ItemHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.update_fail"));
        }

        $update = $request->only(['name_en', 'name_ar', 'account_tree_id','location','item_number', 'category_id', 'item_type_id', 'big_unit_id',
                                  'big_unit_coefficient', 'mid_unit_id', 'mid_unit_coefficient', 'small_unit_id',
                                  'small_unit_coefficient', 'default_report_unit_type', 'sale_price', 'min_sale_price',
                                  'purchases_revenue_id', 'purchases_return_id', 'purchases_discount_id', 'sales_revenue_id',
                                  'sales_return_id','is_service', 'sales_discount_id', 'tax_id', 'time_blocks', 'price_include_vat', 'item_width_height' , 'pos_item' , 'description']);

        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('items')->ignore($id,'item_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('items')->ignore($id,'item_id'),
            ]
            // ,
            // 'item_number' => [
            //     'required',
            //     'string',
            //     Rule::unique('items')->ignore($id,'item_id'),
            // ]
        ];
        $validator  = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        if(isset($request['image']) && strlen($request['image']) > 100){
            $upload = new Upload;
            $check  = $upload->uploadBase64Image($request['image'], 'items_images');
            if(!$check['check'])
                return $this->sendError( $check['msg'] , [] );
            else
                $update['image'] = $check['url'];
        }

        Item::update_item($update, $id);
        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-004','delete');

        if(!(new ItemHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        Item::delete_item($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = Item::find($id);

        if($item->item_active){
            $this->checkPermission('03-004','delete');


            if( !(new ItemHelper)->can_edit_or_delete($id)){
                return $this->sendError( trans('error.delete_fails'), []);
            }

            Item::delete_item($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-004','edit');


            if(!(new ItemHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            Item::update_item(array('item_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
