<?php

namespace App\Http\Controllers\API\V1\Inventory;

use App\Classes\General\Upload;
use App\Classes\ItemStock;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;
use App\Models\Elevators\ElevatorsQuotation;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\ScreenSub;
use App\Models\Inventory\WarehouseTransaction;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
class WarehouseTransactionController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-014','view');

        $this->filter = array(
            'lang'          => $lang ,
            'active'        => $request['active'] ?? 1 ,
            'rows'          => $request['rows'] ?? null ,
            'page'          => $request['page'] ?? 1 ,
            'word'          => $request['word'] ?? null
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-014', $lang)->sub_title ?? '';
        $this->return['items']       = WarehouseTransaction::all_warehouse_trans($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-014','add');
        //check if have elevators module
        $module_ids = DB::table('screen_subs')
        ->where(DB::raw('SUBSTRING(screen_code, 1, 2)'), '=', '12')
        ->pluck('screen_sub_id')->toArray();
        $permissions = DB::table('user_permissions')
        ->leftjoin('screen_subs' , 'user_permissions.screen_sub_id' , '=' , 'screen_subs.screen_sub_id')
        ->where('user_permissions.user_id' , auth()->user()->id)
        ->whereIn('user_permissions.screen_sub_id' , $module_ids)
        ->where('user_permissions.permission' , "<>" , "[]")->count();
        if($permissions > 0){
            $this->return['contract_input'] = true;
            $this->return['contracts'] = ElevatorsQuotation::quotations_lookups($lang);
        }else{
            $this->return['contract_input'] = false;

        }

        $this->return['screen_name']        = ScreenSub::get_screen_sub_by_screen_code('03-014', $lang)->sub_title ?? '';
        $this->return['warehouses']         = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['types']              = WarehouseTransaction::get_types($lang);

        $this->return['items']              = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']  = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-014','add');

        $inputs             = $request->all();
        $insert =[
            'action_date'           => $inputs['action_date'] ?? date('Y-m-d'),
            'contract_id'           => $inputs['contract_id'] ?? 0,
            'description'           => $inputs['description'] ?? null,
            'type_id'               => $inputs['type_id'] ?? null,
        ];
        $valid_contract_id = DB::table('elevators_quotations')->where('invoice_code',$insert['contract_id'])->first();
        if(!$valid_contract_id){
            return $this->sendError(trans('error.contract_not_found') , $insert);

        }
        $validator = validator()->make($insert, WarehouseTransaction::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }
        DB::beginTransaction();
        $this->return['id'] = WarehouseTransaction::add_warehouse_trans_m($insert);
        if(isset($inputs['items']) && is_array($inputs['items'])){
            foreach($inputs['items'] as $item){
                $insert_details = [
                    'warehouse_trans_m_id'  => $this->return['id'],
                    'unit_id'               => $item['unit_id'],
                    'item_id'               => $item['item_id'],
                    'quantity'              => $item['quantity'] ?? 1,
                    'warehouse_id'          => $item['warehouse_id'] ,
                    'amount'                => $item['amount'] ?? 0,
                ];
                $d_id = WarehouseTransaction::add_warehouse_trans_d($insert_details);
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }else{
                        $item['unit_type'] = 3;
                    }
                }else{
                    return $this->sendError(trans('error.unit_required'));
                }
                $type = DB::table('warehouse_trans_types')->where('id',$insert['type_id'])->first();
                $m_id = $this->return['id'];
                $source = ItemStock::Warehouse_transaction_source;

                if ($type->type == 1) {
                    $qty = $item['quantity'];
                } elseif ($type->type == 2) {
                    $qty = (-1 * $item['quantity']);
                }
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $m_id, $d_id, $qty, 0 , $item['unit_id'], $item['unit_type'], 0);

            }
        }
        DB::commit();
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-014','view');
        $module_ids = DB::table('screen_subs')
        ->where(DB::raw('SUBSTRING(screen_code, 1, 2)'), '=', '12')
        ->pluck('screen_sub_id')->toArray();
        $permissions = DB::table('user_permissions')
        ->leftjoin('screen_subs' , 'user_permissions.screen_sub_id' , '=' , 'screen_subs.screen_sub_id')
        ->where('user_permissions.user_id' , auth()->user()->id)
        ->whereIn('user_permissions.screen_sub_id' , $module_ids)
        ->where('user_permissions.permission' , "<>" , "[]")->count();
        if($permissions > 0){
            $this->return['contract_input'] = true;
            $this->return['contracts'] = ElevatorsQuotation::quotations_lookups($lang);

        }else{
            $this->return['contract_input'] = false;

        }
        $this->return['screen_name']        = ScreenSub::get_screen_sub_by_screen_code('03-014', $lang)->sub_title ?? '';
        $this->return['master']             = WarehouseTransaction::get_warehouse_trans_m($id, $lang);
        $this->return['details']            = WarehouseTransaction::get_warehouse_trans_d($id, $lang);
        $this->return['warehouses']         = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']              = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['types']              = WarehouseTransaction::get_types($lang);
        $this->return['unit_measurements']  = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['attachments']        = WarehouseTransaction::get_attachments($id, $lang);

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-014','edit');

        $inputs    = $request->all();
        $update =[
            'action_date'           => $inputs['action_date'] ?? date('Y-m-d'),
            'contract_id'           => $inputs['contract_id'] ?? 0 ,
            'description'           => $inputs['description'] ?? null,
            'type_id'               => $inputs['type_id'] ?? null,
        ];
        $type = DB::table('warehouse_trans_types')->where('id',$update['type_id'])->first();

        $valid_contract_id = DB::table('elevators_quotations')->where('invoice_code',$update['contract_id'])->first();
        if(!$valid_contract_id){
            return $this->sendError(trans('error.contract_not_found'));

        }
        $validator = validator()->make($update, WarehouseTransaction::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }
        DB::beginTransaction();

        WarehouseTransaction::update_warehouse_trans($update, $id);
        // Delete old details
        DB::table('warehouse_trans_d')->where('warehouse_trans_m_id' , $id)->delete();
        // Insert new items updated
        $source = ItemStock::Warehouse_transaction_source;
        if(isset($inputs['items']) && is_array($inputs['items'])){    
            foreach($inputs['items'] as $item){
                $insert_details = [
                    'warehouse_trans_m_id'  => $id,
                    'unit_id'               => $item['unit_id'],
                    'item_id'               => $item['item_id'],
                    'quantity'              => $item['quantity'] ?? 1,
                    'warehouse_id'          => $item['warehouse_id'] ,
                    'amount'                => $item['amount'] ?? 0,
                ];
                $d_id = WarehouseTransaction::add_warehouse_trans_d($insert_details);
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }else{
                        $item['unit_type'] = 3;
                    }
                }else{
                    return $this->sendError(trans('error.unit_required'));
                }


                if ($type->type == 1) {
                    $qty = $item['quantity'];
                } elseif ($type->type == 2) {
                    $qty = (-1 * $item['quantity']);
                }
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $id, $d_id, $qty, 0 , $item['unit_id'], $item['unit_type'], 0);
            }
        }
        DB::commit();

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-014','delete');

        WarehouseTransaction::delete_warehouse_trans($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = WarehouseTransaction::find($id);

        if($item->active){
            $this->checkPermission('03-014','delete');

            WarehouseTransaction::delete_warehouse_trans($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-014','edit');

            WarehouseTransaction::update_warehouse_trans(array('active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function add_attachments(Request $request)
    {
        $this->checkPermission('06-021','add');

        $input    = $request->all();

        $upload = (new Upload)->uploadFile($input['file_url'], 'warehouse_trans_attachments');
                if(!$upload)
                    return $this->sendError(trans("error.Something Error"));
                    $this->return['id'] = DB::table('warehouse_trans_attachments')->insert([
                    'warehouse_trans_m_id'  => $input['warehouse_trans_m_id'],
                    'file_name'         => $inputs['filename'] ?? date('Y-m-d H:i:s'),
                    'file_url'          => $upload['url'],
                ]);

        return $this->sendResponse( $this->return , trans('main.add_success') );
    }
    public function delete_attachments($lang , $id)
    {
        $this->checkPermission('06-021','delete');

        $file_url = DB::table('warehouse_trans_attachments')->where('attachment_id',$id)->value('file_url');
        //delete the file from the server
        if ($file_url && File::exists(public_path($file_url))) {
            File::delete(public_path($file_url));
        }
        WarehouseTransaction::delete_attachments($id);
        return $this->sendResponse( $this->return , trans('main.delete_success') );
    }
    public function warehouse_tansaction_report($lang, Request $request)
    {
        $this->checkPermission('03-014','view');

        $this->filter = array(
            'lang'              => $lang ,
            'active'            => $request['active'] ?? 1 ,
            'from_date'         => $request['from_date'] ?? null ,
            'to_date'           => $request['to_date'] ?? null ,
            'customer_id'       => $request['customer_id'] ?? null ,
            'contract_number'   => $request['contract_number'] ?? null ,
        );

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('03-014', $lang)->sub_title ?? '';
        $this->return['report']         = WarehouseTransaction::all_warehouse_trans($this->filter);

        return $this->sendResponse($this->return);
    }
}
