<?php

namespace App\Http\Controllers\API\V1\POS;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\Item;
use App\Models\Main\Platforms;
use Illuminate\Http\Request;
use App\Models\POS\Coupons;
use App\Models\Main\ScreenSub;
use App\Models\Sales\Customer;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;



class CouponController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang , Request $request)
    {
        $this->checkPermission('06-021','view');
        $this->filter = array(
            'lang'        => $lang ,
            'active'      => $request['active'] ?? 1 ,
            'rows'        => $request['rows'] ?? null ,
            'page'        => $request['page'] ?? 1 ,
            'word'        => $request['word'] ?? null
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['items']       = Coupons::all_coupons($this->filter);

        return $this->sendResponse( $this->return );
    }

    public function create($lang , Request $request)
    {
        $this->checkPermission('06-021','add');

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['services']       = Item::all_items(array('lang' => $lang , 'active' => 1 ));
        $this->return['coupons']        = Coupons::all_codes();
        $this->return['customers']      = Customer::all_customers(array('lang' => $lang , 'active' => 1 ));
        $this->return['platforms']      = Platforms::all_platforms(array('lang' => $lang , 'active' => 1 ));

        return $this->sendResponse( $this->return);
    }

    public function store(Request $request)
    {
        $this->checkPermission('06-021','add');

        $input    = $request->all();
        $insert = [
            'services'      => implode(',',$input['services']),
            'coupon_code'   => $input['coupon_code'],
            'start_date'    => $input['start_date'] ?? date('Y-m-d'),
            'end_date'      => $input['end_date'],
            'coupon_type'   => $input['coupon_type'] ?? 1,
            'discount'      => $input['discount'] ?? 0,
            'platforms_id'  => implode(',',$input['platforms_id']) ?? null,
            'for_customer'  => $input['for_customer'] ?? 0,
            'customers_id'  => implode(',',$input['customers_id']) ?? null,
            'allowed_count' => $input['allowed_count'] ?? 0,
            'add_user'      => auth()->user()->id,
        ];
        // $validator = validator()->make( $insert , Coupons::$rules );
        // if($validator->fails())
        // {
        //     return $this->sendError( trans('error.add_fails') , $validator->errors() );
        // }

        $this->return['id'] = Coupons::add_coupon($insert);

        return $this->sendResponse( $this->return , trans('main.add_success') );
    }

    public function show($lang , $id )
    {
        $this->checkPermission('06-021','view');

        $this->return['screen_name']     = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['item']            = Coupons::get_coupon($id,$lang);
        $this->return['coupons']         = Coupons::all_codes($id);
        $this->return['services']        = Item::all_items(array('lang' => $lang , 'active' => 1 ));
        $this->return['customers']       = Customer::all_customers(array('lang' => $lang , 'active' => 1 ));
        $this->return['platforms']       = Platforms::all_platforms(array('lang' => $lang , 'active' => 1 ));

        return $this->sendResponse( $this->return , '' );
    }

    public function update($lang, $id , Request $request)
    {
        $this->checkPermission('06-021','edit');

        $input  = $request->all();
        $update = [
            'services'      => implode(',',$input['services']),
            'start_date'    => $input['start_date'],
            'end_date'      => $input['end_date'],
            'coupon_type'   => $input['coupon_type'],
            'discount'      => $input['discount'] ,
            'platforms_id'  => implode(',',$input['platforms_id']),
            'for_customer'  => $input['for_customer'] ,
            'customers_id'  => implode(',',$input['customers_id']) ,
            'allowed_count' => $input['allowed_count'] ,
            'coupon_code'   => $input['coupon_code'] ,
        ];
        $update_rules = [
            'coupon_code' => [
                'required',
                'string',
                Rule::unique('coupons')->ignore($id,'coupon_id'),
            ]
        ];
        $validator  = validator()->make( $update , $update_rules );
        if($validator->fails())
        {
            return $this->sendError( trans('error.add_fails') , $validator->errors() );
        }

        Coupons::update_coupon($update ,$id );
        $this->return = Coupons::get_coupon($id,$lang);
        return $this->sendResponse( [] , trans('main.update_success') );
    }
    public function check_code($lang , Request $request)
    {
        // $this->checkPermission('06-021','view');

        $coupon = DB::table('coupons')
                ->where('coupon_code',$request->coupon_code)
                ->first();
        if($coupon == null){
            return $this->sendError(trans('error.code_not_found'), []);
        }
        if($coupon->start_date > date('Y-m-d') && $coupon->end_date < date('Y-m-d')){
            return $this->sendError(trans('error.coupon_date_expired'), []);
        }
        $customers = array_map('intval', explode(',', $coupon->customers_id));
        if($coupon->for_customer == 0){
            $customers = DB::table('customers')->where('customer_active',1)->pluck('customer_id')->toArray();
        }
        $used_coupons = DB::table('sales_invoice_m')
                        ->where([
                            ['sales_invoice_m.customer_id',$request->customer_id],
                            ['sales_invoice_m.coupon_id',$coupon->coupon_id]
                            ])->count();

        if(!in_array($request->customer_id,$customers)){
            return $this->sendError(trans('error.coupon_not_for_this_customer'), []);
        }
        if($coupon->allowed_count < $used_coupons){
            return $this->sendError(trans('error.used_before'), []);
        }

        $this->return['coupon'] = [
            'coupon_id' => $coupon->coupon_id,
            'discount_type' => $coupon->coupon_type ,
             'discount' => $coupon->discount,
             'items'    =>array_map('intval', explode(',', $coupon->services)),
            ];

        return $this->sendResponse($this->return);
    }

    public function destroy($id)
    {
        $this->checkPermission('06-021','delete');

        Coupons::hard_delete_coupon($id);

        return $this->sendResponse( [] , trans('main.delete_success') );
    }


    public function toggle_active($lang,$id){
        $item = Coupons::get_coupon($id,$lang);
        if($item->active){

            $this->checkPermission('06-021','delete');

            Coupons::delete_coupon($id);
            $this->return['active'] = false;

        }else{

            $this->checkPermission('06-021','edit');

            Coupons::update_coupon(array('active' => 1 ) , $id);
            $this->return['active'] = true;
        }
        return $this->sendResponse( $this->return , trans('main.update_success') );
    }


    public function get_coupon_data($lang,$coupon_code)
    {
        $this->checkPermission('06-021','view');
        app()->setLocale($lang);
        $coupon = DB::table('coupons')
            ->where('coupon_code', $coupon_code)
            ->select('coupon_id', 'coupon_code', 'coupon_type as discount_type', 'discount', 'services', 'allowed_count', 'start_date', 'end_date', 'platforms_id')
            ->first();

        if (!$coupon) {
            return $this->sendError(trans('error.Coupon Not Found'));
        }

        // check platform
        if (!in_array(1, explode(',', $coupon->platforms_id))) {
            return $this->sendError(trans('error.Coupon Not Valid For This Platform'));
        }

        // check start date
        if ($coupon->start_date > date("Y-m-d")) {
            return $this->sendError(trans('error.Coupon Not Started Yet'));
        }

        // check end date
        if ($coupon->end_date < date("Y-m-d")) {
            return $this->sendError(trans('error.Coupon Expired'));
        }

        // check if coupon is used
        $counter = DB::table('barber_bookings')
            ->where('barber_bookings.barber_customer_id', auth()->user()->id)
            ->where('coupon_code', $coupon_code)
            ->count();

        if ($counter >= $coupon->allowed_count) {
            return $this->sendError(trans('error.Coupon used before'));
        }
        $coupon->services = array_filter(explode(',', $coupon->services));
        return $this->sendResponse($coupon);
    }

}
