<?php
namespace App\Http\Controllers\API\V1\Purchases;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;

use App\Classes\Excel\InvoicesExport;
use App\Classes\General\Upload;
use App\Classes\ItemStock;
use App\Classes\Purchases\PurchasesInvoiceHelper;

use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Purchases\PurchasesAgent;
use App\Models\Purchases\PurchasesInvoice;
use App\Models\Purchases\PurchasesNote;
use App\Models\Purchases\PurchasesOrder;
use App\Models\Purchases\PurchasesVendor;
use App\Models\Finance\DailyJournal;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Classes\General\General;
use App\Models\Finance\Currency;
use PDF;
use Excel;
use Illuminate\Support\Facades\DB as FacadesDB;
use Storage;

class PurchasesInvoiceController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {

        $this->checkPermission('04-006' , 'view');
        $this->filter = array(
            'lang'         => $lang ,
            'date'         => $request['date'] ?? null,
            'type'         => $request['type'] ?? null,
            'active'       => $request['active'] ?? 1 ,
            'status'       => $request['status'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'customer'     => $request['customer'] ?? null ,
            'branch'       => $this->data['current_branch'] ,
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name']        = ScreenSub::get_screen_sub_by_screen_code('04-006', $lang)->sub_title ?? '';
        $this->return['purchases_invoices'] = PurchasesInvoice::all_purchases_invoices($this->filter);
        $this->return['purchases_orders']   = PurchasesOrder::all_purchases_orders(array('active' => 1, 'status' => 2,'lang' => $lang, 'rows' => $request['rows'] ?? 20, 'page' => $request['page'] ?? 1, 'word' => $request['word'] ?? null));
        $this->return['purchases_notes']    = PurchasesNote::all_purchases_notes($this->filter);
        $this->return['purchases_vendors']  = PurchasesVendor::all_purchases_vendors($this->filter);

        // if(request()->header('clientID') == '3001'){
        //     $x = PurchasesInvoice::all_purchases_invoices(['active' => 1 , 'lang' => 'en' , 'branch' => $this->data['current_branch'] ]);
        //     foreach ($x as $key => $value) {
        //         $m_id = $value->m_id;
        //         (new PurchasesInvoiceHelper)->generate_daily_journal($m_id);
        //     }
        // }
        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('04-006','add');


        $this->return['screen_name']             = ScreenSub::get_screen_sub_by_screen_code('04-006' , $lang)->sub_title ?? '';
        $this->return['purchases_vendors']       = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_agents']        = PurchasesAgent::all_purchases_agents(array('active' => 1, 'lang' => $lang));
        // $this->return['currencys']               = DB::table('currencys')->select('currency_id as id', 'name_'.$lang.' as name')->get();
        $this->return['currencies']              = Currency::all_currency(array('active'=>1, 'lang' => $lang));
        $this->return['warehouses']              = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']                   = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['unit_measurements']       = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']                   = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_invoice_types'] = DB::table('purchases_invoice_types')->select('purchases_invoice_type_id as id', 'purchases_invoice_type_title_'.$lang.' as title')->get();
        $this->return['receiving_locations']     = DB::table('receiving_locations')->select('receiving_location_id as id', 'name_'.$lang.' as name')->get();
        $extra_inputs = (new General)->get_extra_inputs(92 , $lang);
        $this->return['master_inputs']     = $extra_inputs['master'];
        $this->return['detail_inputs']     = $extra_inputs['detail'];
        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('04-006', 'add');


        $insert = $request->only(['purchases_invoice_detail', 'purchases_invoice_date', 'purchases_invoice_type_id', 'purchases_vendor_id','warehouse_id',
            'purchases_agent_id','extra_inputs', 'purchases_vendor_invoice_no', 'purchases_order_id', 'purchases_invoice_description', 'currency_id','discount' ,'discount_type' ,'invoice_discount', 'items']);

        $year   = array_search(date("Y", strtotime($insert['purchases_invoice_date'])), $this->data['years']);

        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        $codes = (new PurchasesInvoiceHelper)->get_next_codes($insert['purchases_invoice_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                     => $this->data['current_branch'],
            'finance_year_id'               => $year,
            'year_ser'                      => $codes['year_ser'],
            'month_ser'                     => $codes['month_ser'],
            'branch_ser'                    => $codes['branch_ser'],
            'daily_journal_m_id'            => 0,
            'extra_inputs'                  => json_encode($insert['extra_inputs'] ?? []),
            'purchases_invoice_detail'      => $insert['purchases_invoice_detail'],
            'purchases_invoice_date'        => $insert['purchases_invoice_date'],
            'purchases_invoice_type_id'     => $insert['purchases_invoice_type_id'],
            'purchases_vendor_id'           => $insert['purchases_vendor_id'],
            'purchases_agent_id'            => $insert['purchases_agent_id'],
            'purchases_vendor_invoice_no'   => $insert['purchases_vendor_invoice_no'],
            'purchases_invoice_description' => $insert['purchases_invoice_description'],
            'purchases_order_id'            => $insert['purchases_order_id'] ?? 0,
            'currency_id'                   => $insert['currency_id'] ?? 1,
            'invoice_discount'              => $insert['invoice_discount'] ?? 0,
            'discount_type'                 => $insert['discount_type'] ?? 0,
            'discount'                      => $insert['discount'] ?? 0,
            'add_user'                      => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, PurchasesInvoice::$master_rules);

        if ($validator->fails()) {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        DB::table('purchases_invoice_m')->lockForUpdate()->get();
        $master_seed['purchases_invoice_m_id'] = intval(DB::table('purchases_invoice_m')->max('purchases_invoice_m_id')) + 1;
        $m_id = PurchasesInvoice::add_purchases_invoice_m($master_seed);
        $source = ItemStock::purchases_source;

        if (isset($insert['items']) && is_array($insert['items'])) {
            foreach ($insert['items'] as $item) {
                $tax_percentage = DB::table('taxes')->where('tax_id', $item['tax_percent_id'])->value('tax_percentage');
                if (isset($item['currency_id']) && $item['currency_id'] > 0) {
                    $equivalent_value = Currency::get_currency($item['currency_id'])->equivalent_value;
                }
                if (isset($item['unit_id']) && !is_null($item['unit_id'])) {
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id) {
                        $item['unit_type'] = 3;
                    } elseif ($item['unit_id'] == $selectedItem->mid_unit_id) {
                        $item['unit_type'] = 2;
                    } elseif ($item['unit_id'] == $selectedItem->small_unit_id) {
                        $item['unit_type'] = 1;
                    }
                }
                $qty = $item['quantity'] ?? 0;
                $details_seed = array(
                    'purchases_invoice_m_id'                => $m_id,
                    'warehouse_id'                          => $item['warehouse_id'],
                    'item_id'                               => $item['item_id'],
                    'unit_id'                               => $item['unit_id'],
                    'unit_type'                             => $item['unit_type'] ?? 0,
                    'quantity'                              => $qty,
                    'price'                                 => $item['price'] ?? 0,
                    'total_price'                           => ($item['price'] * $qty) ?? 0,

                    'currency_id'                           => $item['currency_id'] ?? 0,
                    'foreign_currency'                      => $item['foreign_currency'] ?? 0,
                    'equivalent_value'                      => $equivalent_value ?? null,

                    'tax_percent_id'                        => $item['tax_percent_id'] ?? 0,
                    'total_vat'                             => $item['total_vat'],
                    'total_without_vat'                     => $item['total_price_without_vat'],
                    'total_with_vat'                        => $item['total_with_vat'],
                    'total_discount'                        => $item['total_discount'] ?? 0,
                    'extra_inputs'                          => json_encode($item['extra_inputs'] ?? []),
                    'current_price'                         => $item['price'] ?? 0,
                    'current_total_price'                   => $item['total_price_without_vat'] ?? 0,
                    'current_total_vat'                     => $item['total_vat'],
                    'current_total_without_vat'             => $item['total_price_without_vat'],
                    'current_total_with_vat'                => $item['total_with_vat'],
                    'current_quantity'                      => $item['quantity'] ?? 0,
                    'add_user'                              => auth()->user()->id,
                    'discount_per'                          => $item['discount_per'] ?? 0,
                );

                $is_taxable = DB::table('purchases_vendors')
                    ->where('purchases_vendor_id', $master_seed['purchases_vendor_id'])
                    ->value('is_taxable');

                if ($is_taxable == 0) {
                    $details_seed['total_vat'] = 0;
                    $details_seed['total_with_vat'] = $details_seed['total_without_vat'];
                    $details_seed['current_total_vat'] = 0;
                    $details_seed['current_total_with_vat'] = $details_seed['total_without_vat'];
                }

                $validator = validator()->make($details_seed, PurchasesInvoice::$details_rules);

                if ($validator->fails()) {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = PurchasesInvoice::add_purchases_invoice_d($details_seed);

                // Check If Request Come From Purchases Note
                if (isset($item['m_id']) && !is_null($item['m_id'])) {
                    // Update Purchases Note With Purchases Invoice ID and Close It
                    DB::table('purchases_note_m')
                        ->where('purchases_note_m_id', $item['m_id'])
                        ->update([
                            'purchases_note_status' => 2,
                            'purchases_invoice_id'  => $m_id
                        ]);
                    // Close Purchases Order
                    DB::table('purchases_order_m')
                        ->where('purchases_order_m_id', $insert['purchases_order_id'])
                        ->update([
                            'purchases_invoice_id' => $m_id
                        ]);
                } else {
                    // Item Stock
                    (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $m_id, $d_id, $qty, 0, $item['unit_id'], $item['unit_type'], 0);
                }
            }
        }

        (new PurchasesInvoiceHelper)->update_master_values($m_id);
        (new PurchasesInvoiceHelper)->generate_daily_journal($m_id);


        DB::commit();
        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('04-006','view');


        $this->return['screen_name']             = ScreenSub::get_screen_sub_by_screen_code('04-006', $lang)->sub_title ?? '';
        $this->return['master']                  = PurchasesInvoice::get_purchases_invoice_m($id, $lang);
        $this->return['details']                 = PurchasesInvoice::get_purchases_invoice_d($id, $lang);
        $this->return['master']->extra_inputs = json_decode($this->return['master']->extra_inputs);
        foreach ($this->return['details'] as $key => $value) {
            $this->return['details'][$key]->extra_inputs = json_decode($this->return['details'][$key]->extra_inputs);
        }
        $this->return['files']                   = PurchasesInvoice::get_purchases_invoice_files($id);

        $this->return['purchases_vendors']       = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_agents']        = PurchasesAgent::all_purchases_agents(array('active' => 1, 'lang' => $lang));
        // $this->return['currencys']               = DB::table('currencys')->select('currency_id as id', 'name_'.$lang.' as name')->get();
        $this->return['currencies']              = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        $this->return['warehouses']              = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']                   = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['unit_measurements']       = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']                   = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_invoice_types'] = DB::table('purchases_invoice_types')->select('purchases_invoice_type_id as id', 'purchases_invoice_type_title_'.$lang.' as title')->get();

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {

        $this->checkPermission('04-006','edit');

        $update = $request->only(['discount','discount_type','invoice_discount','purchases_invoice_date','extra_inputs', 'purchases_invoice_type_id', 'purchases_invoice_detail', 'purchases_vendor_id', 'purchases_agent_id', 'purchases_vendor_invoice_no', 'currency_id', 'purchases_invoice_description']);
        $update['extra_inputs'] = json_encode($update['extra_inputs'] ?? [] );
        $m_id = $id ;

        DB::beginTransaction();

            PurchasesInvoice::update_purchases_invoice_m($update, $id);
            $old        = PurchasesInvoice::get_purchases_invoice_d($id , $lang);
            $new_ids    = collect($request['items'])->pluck('d_id')->toArray();
            $source = ItemStock::purchases_source;

            foreach ($old as $row){
                if( !in_array( $row->d_id , $new_ids ) ){
                    PurchasesInvoice::delete_purchases_invoice_d($row->d_id );
                    (new ItemStock)->update_item_stock($this->data['current_branch'],  $row->item_id, $row->warehouse_id, $source, $m_id, $row->d_id, ( $row->quantity * -1 ), 0, $row->unit_id, $row->unit_type ?? 0, 0);
                }
            }

            foreach ($request['items'] as $item){
                $tax_percentage = DB::table('taxes')->where('tax_id', $item['tax_percent_id'])->value('tax_percentage');
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }
                }
                if(isset($item['currency_id']) && $item['currency_id'] > 0){
                    $equivalent_value = Currency::get_currency($item['currency_id'])->equivalent_value;
                }


                $details_seed = array(
                    'purchases_invoice_m_id'    => $m_id,
                    'warehouse_id'              => $item['warehouse_id'],
                    'item_id'                   => $item['item_id'],
                    'unit_id'                   => $item['unit_id'],
                    'unit_type'                 => $item['unit_type'] ?? 0,
                    'quantity'                  => $item['quantity'] ?? 0,
                    'price'                     => $item['price'] ?? 0,
                    'total_price'               => $item['total_price_without_vat'] ,

                    'currency_id'               => $item['currency_id'] ?? 0,
                    'foreign_currency'          => $item['foreign_currency'] ?? 0,
                    'equivalent_value'          => $equivalent_value ?? null,

                    'tax_percent_id'            => $item['tax_percent_id'] ?? 0,
                    'total_vat'                 => $item['total_vat'] ,
                    'total_without_vat'         => $item['total_price_without_vat'] ,
                    'total_with_vat'            => $item['total_with_vat'] ,
                    'total_discount'            => $item['total_discount'] ?? 0 ,
                    'extra_inputs'              => json_encode($item['extra_inputs'] ?? [] ),
                    'current_price'             => $item['price'] ?? 0 ,
                    'current_total_price'       => $item['total_price_without_vat'] ?? 0 ,
                    'current_total_vat'         => $item['total_vat'] ,
                    'current_total_without_vat' => $item['total_price_without_vat'] ,
                    'current_total_with_vat'    => $item['total_with_vat'] ,
                    'current_quantity'          => $item['quantity'] ?? 0,
                    'add_user'                  => auth()->user()->id, 
                    'discount_per'              => $item['discount_per'] ?? 0
                );


                $is_taxable = DB::table('purchases_vendors')
                    ->where('purchases_vendor_id',$update['purchases_vendor_id'])
                    ->value('is_taxable');
                    if($is_taxable == 0){
                        $details_seed['total_vat'] = 0;
                        $details_seed['total_with_vat'] = $details_seed['total_without_vat'];
                    }
                $validator = validator()->make($details_seed, PurchasesInvoice::$details_rules);

                if($validator->fails())
                {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }



                if(!isset($item['d_id']) || $item['d_id'] == 0 ){
                    $d_id = PurchasesInvoice::add_purchases_invoice_d($details_seed);
                    (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $m_id, $d_id, $item['quantity'], 0, $item['unit_id'], $item['unit_type'], 0);

                }else{
                    $check = DB::table('purchases_invoice_d')
                        ->where('purchases_invoice_d_id' , $item['d_id'])
                        ->first();
                    if($check->current_quantity < $check->quantity ){
                        continue;
                    }
                    unset($details_seed['add_user']);

                    PurchasesInvoice::update_purchases_invoice_d($details_seed , $item['d_id'] );

                }

            }

            (new PurchasesInvoiceHelper)->update_master_values($m_id);
            (new PurchasesInvoiceHelper)->generate_daily_journal($m_id);

        DB::commit();



        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('04-006','delete');

        $master = PurchasesInvoice::get_purchases_invoice_m($id);

        if(!isset($master->purchases_invoice_m_id) || $master->branch_id != $this->data['current_branch'] || !(new PurchasesInvoiceHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        DB::beginTransaction();
            PurchasesInvoice::delete_purchases_invoice_m($id);
            DailyJournal::delete_daily_journal_m( $master->daily_journal_m_id );
        DB::commit();


        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = PurchasesInvoice::find($id);

        if($item->purchases_invoice_active){
            $this->checkPermission('04-006','delete');

            if(!(new PurchasesInvoiceHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            DB::beginTransaction();
                PurchasesInvoice::delete_purchases_invoice_m($id);
                DailyJournal::delete_daily_journal_m( $item->daily_journal_m_id );
            DB::commit();


            $this->return['active'] = false;
        }else{
            $this->checkPermission('04-006','edit');


            if(!(new PurchasesInvoiceHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            DB::beginTransaction();
                PurchasesInvoice::update_purchases_invoice_m(array('purchases_invoice_active' => 1), $id);
                (new PurchasesInvoiceHelper)->generate_daily_journal($id);

            DB::commit();

            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function get_purchases_note_data($lang, Request $request)
    {
        if (isset($request['purchases_notes']) && !is_null($request['purchases_notes']) && is_array($request['purchases_notes'])){
            $purchases_notes = DB::table('purchases_note_m')
                ->whereIn('purchases_note_m_id', $request['purchases_notes'])
                ->pluck('purchases_note_m_id');
        }else{
            $purchases_notes = DB::table('purchases_note_m')
                ->where('purchases_order_id', $request['purchases_order_id'])
                ->pluck('purchases_note_m_id');
        }

        $query = DB::table('purchases_note_d')
            ->leftjoin('items' , 'purchases_note_d.item_id' , '=' , 'items.item_id')
            ->leftjoin('warehouses' , 'purchases_note_d.warehouse_id' , '=' , 'warehouses.warehouse_id')
            ->leftjoin('unit_measurements' , 'purchases_note_d.unit_id' , '=' , 'unit_measurements.unit_measurement_id')
            ->leftjoin('taxes' , 'purchases_note_d.tax_percent_id' , '=' , 'taxes.tax_id')
            ->whereIn('purchases_note_m_id', $purchases_notes);

        if($lang != null){
            $query->selectRaw('
                purchases_note_m_id as m_id ,
                purchases_note_d_id as d_id ,
                purchases_note_d.item_id ,
                items.name_'.$lang.' as item_name ,
                purchases_note_d.unit_id ,
                unit_measurements.name_'.$lang.' as unit_name ,
                purchases_note_d.warehouse_id ,
                warehouses.name_'.$lang.' as warehouse_name ,
                purchases_note_d.quantity ,
                purchases_note_d.price ,
                purchases_note_d.total_price ,
                purchases_note_d.tax_percent_id ,
                taxes.tax_percentage as tax_percent ,
                purchases_note_d.total_vat ,
                purchases_note_d.total_without_vat ,
                purchases_note_d.total_with_vat
            ');
        }

        $this->return['items'] = $query->get();
        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function print_invoice($id, Request $request)
    {
        $input = $request->all();

        App::setLocale('ar');

        $this->data['lang']        = 'ar';
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-008' , $this->data['lang']);
        $this->data['master']      = PurchasesInvoice::get_purchases_invoice_m($id, $this->data['lang']);
        $this->data['details']     = PurchasesInvoice::get_purchases_invoice_d($id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);

        $temp_name                 = Setting::get_main_setting_by_code('purchases_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Purchases.purchases_invoice.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }

    public function add_attach($lang , $id , Request $request){

        $upload = new Upload;
        $check  = $upload->uploadFile($request['file'] , 'purchase_attachments' );
        if(!$check['check'])
            return $this->sendError( trans('error.add_fails') , $check['msg'] );

        $this->return['id'] = DB::table('purchases_invoice_attachments')->insertGetId( [
            'purchase_invoice_id'   => $id ,
            'file_name'             => $request['file_name']  ,
            'file_path'             => $check['url']  ,
            'file_ext'              => $check['extension']  ,
            'add_user'              => auth()->user()->id
        ]);

        return $this->sendResponse( $this->return , '' );
    }

    public function download_attach($lang , $id){

        $file = DB::table('purchases_invoice_attachments')
            ->where('attachment_id' , $id)
            ->first();

        return response()->file(public_path().$file->file_path);

    }

    public function get_attachs($lang , $id){
        $this->return['data'] = PurchasesInvoice::get_purchases_invoice_files($id);
        return $this->sendResponse( $this->return , '' );

    }

    public function delete_attach($lang , $id){

        $file = DB::table('purchases_invoice_attachments')
            ->where('attachment_id' , $id)
            ->first();

        DB::table('purchases_invoice_attachments')
            ->where('attachment_id' , $id)
            ->delete();

        if(Storage::exists($file->file_path)){
            Storage::delete($file->file_path);
        }

        return $this->sendResponse( $this->return , '' );
    }


}
