<?php

namespace App\Http\Controllers\API\V1\Purchases;
use App\Classes\Excel\InvoicesExport;
use App\Classes\General\Upload;
use App\Classes\ItemStock;
use App\Classes\Purchases\PurchasesNoteHelper;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Purchases\PurchasesNote;
use App\Models\Purchases\PurchasesOrder;
use App\Models\Purchases\PurchasesVendor;
use Illuminate\Http\Request;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Models\Finance\Currency;
use PDF;
use Excel;

class PurchasesNoteController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('04-005','view');

        $this->filter = array(
            'lang'         => $lang ,
            'active'       => $request['active'] ?? 1 ,
            'status'       => $request['status'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'branch'       => $this->data['current_branch'] ,
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name']               = ScreenSub::get_screen_sub_by_screen_code('04-005', $lang)->sub_title ?? '';
        $this->return['purchases_notes']           = PurchasesNote::all_purchases_notes($this->filter);
        $this->return['completed_purchases_notes'] = PurchasesNote::all_purchases_notes(array('active' => 1, 'status' => 2,'lang' => $lang, 'rows' => $request['rows'] ?? 20, 'page' => $request['page'] ?? 1, 'word' => $request['word'] ?? null));
        $this->return['purchases_orders']          = PurchasesOrder::all_purchases_orders(array('active' => 1, 'status' => 2,'lang' => $lang, 'rows' => $request['rows'] ?? 20, 'page' => $request['page'] ?? 1, 'word' => $request['word'] ?? null));

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('04-005','add');


        $this->return['screen_name']         = ScreenSub::get_screen_sub_by_screen_code('04-005' , $lang)->sub_title ?? '';
        $this->return['purchases_vendors']   = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
//        $this->return['purchases_orders']    = PurchasesOrder::all_purchases_orders(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_orders']    = DB::table('purchases_order_m')->where('purchases_order_status', 2)->select('purchases_order_m_id as m_id')->get();
        $this->return['payment_methods']     = DB::table('payment_methods')->select('payment_method_id as id', 'name_'.$lang.' as name')->get();
        $this->return['warehouses']          = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']               = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']   = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']               = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['receiving_locations'] = DB::table('receiving_locations')->select('receiving_location_id', 'name_'.$lang.' as name')->get();
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('04-005','add');


        $insert = $request->only(['purchases_note_date', 'purchases_note_no', 'purchases_vendor_id', 'purchases_order_id',
                                  'purchases_vendor_invoice_no', 'purchases_note_description', 'items']);

        $year   = array_search(date("Y", strtotime($insert['purchases_note_date'])), $this->data['years']);

        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        $codes = (new PurchasesNoteHelper)->get_next_codes($insert['purchases_note_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                   => $this->data['current_branch'],
            'finance_year_id'             => $year,
            'year_ser'                    => $codes['year_ser'],
            'month_ser'                   => $codes['month_ser'],
            'branch_ser'                  => $codes['branch_ser'],
            'daily_journal_m_id'          => 0,
            'purchases_note_date'         => $insert['purchases_note_date'],
            'purchases_note_no'           => $insert['purchases_note_no'],
            'purchases_vendor_id'         => $insert['purchases_vendor_id'],
            'purchases_order_id'          => $insert['purchases_order_id'] ?? null,
            'purchases_vendor_invoice_no' => $insert['purchases_vendor_invoice_no'],
            'purchases_note_description'  => $insert['purchases_note_description'],
            'add_user'                    => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, PurchasesNote::$master_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $m_id = PurchasesNote::add_purchases_note_m($master_seed);
        $source = ItemStock::Purchases_Note_source;
        if (isset($insert['items']) && is_array($insert['items'])){
            foreach ($insert['items'] as $item){

                if (isset($item['received_quantity']) && !is_null($item['received_quantity'])){
                    if($item['received_quantity'] == 0 || $item['received_quantity'] < 0){
                        return $this->sendError(trans("error.You Must Put Real Value In Received Quantity."));
                    }

                    if($item['received_quantity'] > $item['quantity']){
                        return $this->sendError(trans("error.Received Quantity must be less than or equal quantity in purchase order."));
                    }
                }

                $tax_percentage = DB::table('taxes')->where('tax_id', $item['tax_percent_id'])->value('tax_percentage');
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }else{
                        $item['unit_type'] = 3;
                    }
                }
                if(isset($item['currency_id']) && $item['currency_id'] > 0){
                    $equivalent_value = Currency::get_currency($item['currency_id'])->equivalent_value;
                }
                $qty = ((isset($item['received_quantity']) && !is_null($item['received_quantity']) && $item['received_quantity'] != 0) ? $item['received_quantity'] : $item['quantity']);

                $details_seed = array(
                    'purchases_note_m_id'   => $m_id,
                    'warehouse_id'          => $item['warehouse_id'],
                    'item_id'               => $item['item_id'],
                    'unit_id'               => $item['unit_id'],
                    'unit_type'             => $item['unit_type'] ?? 0,
                    'quantity'              => $qty,
                    'price'                 => $item['price'],

                    'currency_id'          => $item['currency_id'] ?? 0,
                    'foreign_currency'     => $item['foreign_currency'] ?? 0,
                    'equivalent_value'     => $equivalent_value ?? null,

                    'total_price'           => ($item['price'] * $qty),
                    'tax_percent_id'        => $item['tax_percent_id'] ?? 0,
                    'total_vat'             => ($item['price'] * $qty) * (($tax_percentage ?? 15) / 100),
                    'total_without_vat'     => ($item['price'] * $qty),
                    'total_with_vat'        => ($item['price'] * $qty) + (($item['price'] * $qty) * (($tax_percentage ?? 15) / 100)),

                    'receiving_location_id' => $item['receiving_location_id'] ?? 0,
                    'add_user'              => auth()->user()->id,

                    // Save Purchases Order Detail ID On Purchase Note Detail
                    'purchases_order_d_id'  => $item['d_id'] ?? 0
                );

                $validator = validator()->make($details_seed, PurchasesNote::$details_rules);

                if($validator->fails())
                {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = PurchasesNote::add_purchases_note_d($details_seed);

                // If Request Come From (Purchases Order) - Update Purchases Order Detail Data
                if (isset($item['d_id']) && !is_null($item['d_id']) && strlen($item['d_id']) > 0){

                    DB::table('purchases_order_d')
                        ->where('purchases_order_d_id', $item['d_id'])
                        ->where('purchases_order_m_id', $item['m_id'])
                        ->increment('received_quantity', $item['received_quantity']);

                    DB::table('purchases_order_d')
                        ->where('purchases_order_d_id', $item['d_id'])
                        ->where('purchases_order_m_id', $item['m_id'])
                        ->decrement('remaining_quantity', ($item['received_quantity'] ?? 0));

                    // Check If Quantities Of Purchases Order Equal To Received Quantities - Close Purchases Order
                    $all_po_details = DB::table('purchases_order_d')->where('purchases_order_m_id', $item['m_id'])->get();
                    if (collect($all_po_details)->sum('quantity') == collect($all_po_details)->sum('received_quantity')){
                        DB::table('purchases_order_m')
                            ->where('purchases_order_m_id', $item['m_id'])
                            ->update([
                                'purchases_order_status' => 4,
                            ]);
                    }
                }
                $source = ItemStock::Purchases_Note_source;
                // Item Stock
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $item['warehouse_id'], $source, $m_id, $d_id, $qty, 0 , $item['unit_id'], $item['unit_type'], 0);
            }
        }

        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'purchases_note_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('purchases_note_files')->insert([
                        'purchases_note_m_id' => $m_id,
                        'purchases_note_d_id' => 0,
                        'file_name'           => $upload['filename'],
                        'file_path'           => $upload['url'],
                        'file_extension'      => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'purchases_note_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('purchases_note_files')->insert([
                'purchases_note_m_id' => $m_id,
                'purchases_note_d_id' => 0,
                'file_name'           => $upload['filename'],
                'file_path'           => $upload['url'],
                'file_extension'      => $upload['extension'],
            ]);
        }

        $updateMasterValues   = (new PurchasesNoteHelper)->update_master_values($m_id);
        $generateDailyJournal = (new PurchasesNoteHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('04-005','view');


        $this->return['screen_name']         = ScreenSub::get_screen_sub_by_screen_code('04-005', $lang)->sub_title ?? '';
        $this->return['master']              = PurchasesNote::get_purchases_note_m($id, $lang);
        $this->return['details']             = PurchasesNote::get_purchases_note_d($id, $lang);
        $this->return['files']               = PurchasesNote::get_purchases_note_files($id);

        $this->return['purchases_vendors']   = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['purchases_orders']    = DB::table('purchases_order_m')->where('purchases_order_status', 2)->select('purchases_order_m_id as m_id')->get();
        $this->return['payment_methods']     = DB::table('payment_methods')->select('payment_method_id as id', 'name_'.$lang.' as name')->get();
        $this->return['warehouses']          = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']               = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']   = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']               = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['receiving_locations'] = DB::table('receiving_locations')->select('receiving_location_id', 'name_'.$lang.' as name')->get();
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('04-005','edit');


        $update = $request->only(['purchases_note_date', 'purchases_note_no', 'purchases_vendor_id', 'purchases_order_id', 'purchases_vendor_invoice_no', 'purchases_note_description']);

        PurchasesNote::update_purchases_note_m($update, $id);
        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('04-005','delete');


        $master = PurchasesNote::get_purchases_note_m($id);

        if(!isset($master->purchases_note_m_id) || $master->branch_id != $this->data['current_branch'] || !(new PurchasesNoteHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        PurchasesNote::delete_purchases_note_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = PurchasesNote::find($id);

        if($item->purchases_note_active){
            $this->checkPermission('04-005','delete');


            if(!(new PurchasesNoteHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            PurchasesNote::delete_purchases_note_m($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('04-005','view');


            if(!(new PurchasesNoteHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            PurchasesNote::update_purchases_note_m(array('purchases_note_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function uploadFiles($lang, Request $request)
    {
        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'purchases_note_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('purchases_note_files')->insert([
                        'purchases_note_m_id' => $request['id'],
                        'purchases_note_d_id' => 0,
                        'file_name'           => $upload['filename'],
                        'file_path'           => $upload['url'],
                        'file_extension'      => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'purchases_note_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('purchases_note_files')->insert([
                'purchases_note_m_id' => $request['id'],
                'purchases_note_d_id' => 0,
                'file_name'           => $upload['filename'],
                'file_path'           => $upload['url'],
                'file_extension'      => $upload['extension'],
            ]);
        }

        return $this->sendResponse([], trans('main.Upload Successfully'));
    }

    public function accept_purchases_note($lang, $id, Request $request)
    {
        $purchase_note = DB::table('purchases_note_m')->where('purchases_note_m_id', $id)->first();

        if ($purchase_note->purchases_note_status == 2){
            return $this->sendError(trans("error.Purchases Note Already Accepted"));
        }

        DB::table('purchases_note_m')
            ->where('purchases_note_m_id', $id)
            ->update([
                'purchases_note_status' => 2
            ]);

        return $this->sendResponse([], trans('main.Purchases Note Accepted Successfully'));
    }

    public function decline_purchases_note($lang, $id, Request $request)
    {
        $purchase_note = DB::table('purchases_note_m')->where('purchases_note_m_id', $id)->first();

        if ($purchase_note->purchases_note_status == 3){
            return $this->sendError(trans("error.Purchases Note Already Declined"));
        }

        DB::table('purchases_note_m')
            ->where('purchases_note_m_id', $id)
            ->update([
                'purchases_note_status' => 3
            ]);

        return $this->sendResponse([], trans('main.Purchases Note Declined Successfully'));
    }

    public function get_vendor_purchases_order($lang, Request $request)
    {
        $this->return['items'] = DB::table('purchases_order_m')
            ->where('purchases_vendor_id', $request['purchases_vendor_id'])
            ->where('purchases_order_status', 2)
            ->select('purchases_order_m_id as id')->get();

        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function get_purchases_order_data($lang, Request $request)
    {
        $this->return['items'] = PurchasesOrder::get_purchases_order_d($request['purchases_order_id'], $lang);
        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function print_invoice($id, Request $request)
    {
        $input = $request->all();

        App::setLocale('ar');

        $this->data['lang']        = 'ar';
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-008' , $this->data['lang']);
        $this->data['master']      = PurchasesNote::get_purchases_note_m($id, $this->data['lang']);
        $this->data['details']     = PurchasesNote::get_purchases_note_d($id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);

        $temp_name                 = Setting::get_main_setting_by_code('purchases_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Purchases.purchases_note.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }
}
