<?php

namespace App\Http\Controllers\API\V1\Purchases;
use App\Classes\Excel\InvoicesExport;
use App\Classes\General\Upload;
use App\Classes\Purchases\PurchasesOrderHelper;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Purchases\PurchasesOrder;
use App\Models\Purchases\PurchasesVendor;
use Illuminate\Http\Request;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Mail\SendMail;
use App\Models\Finance\Currency;
use PDF;
use Excel;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Mail;

class PurchasesOrderController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('04-004', 'view');

        $this->filter = array(
            'lang'         => $lang ,
            'active'       => $request['active'] ?? 1 ,
            'status'       => $request['status'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'branch'       => $this->data['current_branch'] ,
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('04-004', $lang)->sub_title ?? '';
        $this->return['items']       = PurchasesOrder::all_purchases_orders($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('04-004', 'add');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('04-004' , $lang)->sub_title ?? '';
        $this->return['purchases_vendors'] = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['payment_methods']   = DB::table('payment_methods')->select('payment_method_id as id', 'name_'.$lang.' as name')->get();
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('04-004', 'add');

        $insert = $request->only(['purchases_order_date', 'purchases_order_no_date', 'purchases_order_description', 'purchases_vendor_id',
                                  'payment_method_id', 'receiving_palace', 'transportation', 'offer_due_term', 'items']);

        $year   = array_search(date("Y", strtotime($insert['purchases_order_date'])), $this->data['years']);

        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        $codes = (new PurchasesOrderHelper)->get_next_codes($insert['purchases_order_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                   => $this->data['current_branch'],
            'finance_year_id'             => $year,
            'year_ser'                    => $codes['year_ser'],
            'month_ser'                   => $codes['month_ser'],
            'branch_ser'                  => $codes['branch_ser'],
            'daily_journal_m_id'          => 0,
            'purchases_order_date'        => $insert['purchases_order_date'],
            'purchases_order_no_date'     => $insert['purchases_order_no_date'],
            'purchases_order_description' => $insert['purchases_order_description'],
            'purchases_vendor_id'         => $insert['purchases_vendor_id'],
            'payment_method_id'           => $insert['payment_method_id'],
            'receiving_palace'            => $insert['receiving_palace'],
            'transportation'              => $insert['transportation'],
            'offer_due_term'              => $insert['offer_due_term'],
            'add_user'                    => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, PurchasesOrder::$master_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $m_id = PurchasesOrder::add_purchases_order_m($master_seed);

        if (isset($insert['items']) && is_array($insert['items'])){
            foreach ($insert['items'] as $item){
                $tax_percentage = DB::table('taxes')
                    ->where('tax_id', $item['tax_percent_id'])
                    ->value('tax_percentage');
                if(isset($item['currency_id']) && $item['currency_id'] > 0){
                    $equivalent_value = Currency::get_currency($item['currency_id'])->equivalent_value;
                }
                if (isset($item['unit_id']) && !is_null($item['unit_id'])){
                    $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                    if ($item['unit_id'] == $selectedItem->big_unit_id){
                        $item['unit_type'] = 3;
                    }elseif ($item['unit_id'] == $selectedItem->mid_unit_id){
                        $item['unit_type'] = 2;
                    }elseif ($item['unit_id'] == $selectedItem->small_unit_id){
                        $item['unit_type'] = 1;
                    }
                }

                $details_seed = array(
                    'purchases_order_m_id' => $m_id,
                    'warehouse_id'         => $item['warehouse_id'],
                    'item_id'              => $item['item_id'],
                    'unit_id'              => $item['unit_id'],
                    'unit_type'            => $item['unit_type'] ?? 0,
                    'quantity'             => $item['quantity'] ?? 0,
                    'price'                => $item['price'] ?? 0,
                    'total_price'          => ($item['price'] * $item['quantity']),

                    'currency_id'          => $item['currency_id'] ?? 0,
                    'foreign_currency'     => $item['foreign_currency'] ?? 0,
                    'equivalent_value'     => $equivalent_value ?? null,

                    'tax_percent_id'       => $item['tax_percent_id'] ?? 0,
                    'total_vat'            => ($item['price'] * $item['quantity']) * (($tax_percentage ?? 15) / 100),
                    'total_without_vat'    => ($item['price'] * $item['quantity']),
                    'total_with_vat'       => ($item['price'] * $item['quantity']) + (($item['price'] * $item['quantity']) * (($tax_percentage ?? 15) / 100)),
                    'add_user'             => auth()->user()->id,

                    'received_quantity'    => $item['received_quantity'] ?? 0,
                    'remaining_quantity'   => $item['quantity'] ?? 0,
                );

                $validator = validator()->make($details_seed, PurchasesOrder::$details_rules);

                if($validator->fails())
                {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = PurchasesOrder::add_purchases_order_d($details_seed);
            }
        }

        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'purchases_order_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('purchases_order_files')->insert([
                        'purchases_order_m_id' => $m_id,
                        'purchases_order_d_id' => 0,
                        'file_name'            => $upload['filename'],
                        'file_path'            => $upload['url'],
                        'file_extension'       => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'purchases_order_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('purchases_order_files')->insert([
                'purchases_order_m_id' => $m_id,
                'purchases_order_d_id' => 0,
                'file_name'            => $upload['filename'],
                'file_path'            => $upload['url'],
                'file_extension'       => $upload['extension'],
            ]);
        }

        $updateMasterValues   = (new PurchasesOrderHelper)->update_master_values($m_id);
        $generateDailyJournal = (new PurchasesOrderHelper)->generate_daily_journal($m_id);
        $encrypted_data = array('id'=>$m_id );
        $link_token = $this->encryptRouteParameter($encrypted_data);
        DB::commit();
        $sendmail = (new PurchasesOrderHelper)->send_to_responsible_users($m_id , $link_token , request()->header('clientID'));

        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('04-004', 'view');

        $this->return['screen_name']            = ScreenSub::get_screen_sub_by_screen_code('04-004', $lang)->sub_title ?? '';
        $this->return['master']                 = PurchasesOrder::get_purchases_order_m($id, $lang);
        $this->return['details']                = PurchasesOrder::get_purchases_order_d($id, $lang);
        $this->return['files']                  = PurchasesOrder::get_purchases_order_files($id);
        $this->return['purchases_vendors']      = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['payment_methods']        = DB::table('payment_methods')->select('payment_method_id as id', 'name_'.$lang.' as name')->get();
        $this->return['warehouses']             = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']                  = Item::all_items(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements']      = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']                  = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['approvement_permission'] = (new PurchasesOrderHelper)->check_approvement_permission();
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('04-004', 'edit');

        $update = $request->only(['purchases_order_date', 'purchases_order_no_date', 'purchases_order_description', 'purchases_vendor_id', 'payment_method_id', 'receiving_palace', 'transportation', 'offer_due_term']);

        PurchasesOrder::update_purchases_order_m($update, $id);
        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('04-004', 'delete');

        $master = PurchasesOrder::get_purchases_order_m($id);

        if(!isset($master->purchases_order_m_id) || $master->branch_id != $this->data['current_branch'] || !(new PurchasesOrderHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        PurchasesOrder::delete_purchases_order_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = PurchasesOrder::find($id);

        if($item->purchases_order_active){
            $this->checkPermission('04-004', 'delete');

            if(!(new PurchasesOrderHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            PurchasesOrder::delete_purchases_order_m($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('04-004', 'edit');

            if(!(new PurchasesOrderHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            PurchasesOrder::update_purchases_order_m(array('purchases_order_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function uploadFiles($lang, Request $request)
    {
        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'purchases_order_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('purchases_order_files')->insert([
                        'purchases_order_m_id' => $request['id'],
                        'purchases_order_d_id' => 0,
                        'file_name'            => $upload['filename'],
                        'file_path'            => $upload['url'],
                        'file_extension'       => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'purchases_order_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('purchases_order_files')->insert([
                'purchases_order_m_id' => $request['id'],
                'purchases_order_d_id' => 0,
                'file_name'            => $upload['filename'],
                'file_path'            => $upload['url'],
                'file_extension'       => $upload['extension'],
            ]);
        }

        return $this->sendResponse([], trans('main.Upload Successfully'));
    }

    public function accept_purchase_order($lang, $id, $encryptedParameter = null)
    {
        if(isset($encryptedParameter)){
            $decryptedData = $this->decryptRouteParameter($encryptedParameter);
            $id = $decryptedData['id'];
        }
        $purchase_order = PurchasesOrder::get_purchases_order_m($id, $lang);

        $added_user_email = DB::table('users')->where('id',$purchase_order->add_user)->value('email');
        if ($purchase_order->purchases_order_status != 1){
            if(isset($encryptedParameter)){
                return view('emails.purchase_order_response', [
                    'response' => "Expired Link"
                ]);
            }else{
                return $this->sendError(trans("error.allredy_had_action"));
            }
        }

        DB::table('purchases_order_m')
            ->where('purchases_order_m_id', $id)
            ->update([
                'purchases_order_status' => 2
            ]);
        Mail::to($added_user_email)->send(new SendMail([
            'response'      => "Purchases Order Accepted Successfully" ,
            'supplier'      => $purchase_order->purchases_vendor_name ,
            'id'            => $id,
            'amount'        => $purchase_order->total_with_vat ,
            'currency'      => "SAR"
        ] , 'emails.purchase_order_response'));

        if(isset($encryptedParameter)){
            return view('emails.purchase_order_response',[
                'response'      => "Purchases Order Accepted Successfully" ,
                'supplier'      => $purchase_order->purchases_vendor_name ,
                'id'            => $id,
                'amount'        => $purchase_order->total_with_vat ,
                'currency'      => "SAR"
            ]);
        }else{
            return $this->sendResponse([], trans('main.Purchases Order Accepted Successfully'));
        }


    }

    public function decline_purchase_order($lang, $id, $encryptedParameter = null)
    {
        if(isset($encryptedParameter)){
            $decryptedData = $this->decryptRouteParameter($encryptedParameter);
            $id = $decryptedData['id'];
        }
        $purchase_order = PurchasesOrder::get_purchases_order_m($id, $lang);
        $added_user_email = DB::table('users')->where('id',$purchase_order->add_user)->value('email');

        if ($purchase_order->purchases_order_status != 1){
            if(isset($encryptedParameter)){
                return view('emails.purchase_order_response', [
                    'response' => "Expired Link"
                ]);
            }else{
                return $this->sendError(trans("error.allredy_had_action"));
            }
        }
        DB::table('purchases_order_m')
            ->where('purchases_order_m_id', $id)
            ->update([
                'purchases_order_status' => 3
            ]);
        Mail::to($added_user_email)->send(new SendMail([
            'response'      => "Purchases Order Declined Successfully" ,
            'supplier'      => $purchase_order->purchases_vendor_name ,
            'id'            => $id,
            'amount'        => $purchase_order->total_with_vat ,
            'currency'      => "SAR"
        ] , 'emails.purchase_order_response'));
        if(isset($encryptedParameter)){
            return view('emails.purchase_order_response', [
                'response'      => "Purchases Order Declined Successfully" ,
                'supplier'      => $purchase_order->purchases_vendor_name ,
                'id'            => $id,
                'amount'        => $purchase_order->total_with_vat ,
                'currency'      => "SAR"
            ]);
        }else{
            return $this->sendResponse([], trans('main.Purchases Order Declined Successfully'));
        }
    }
    public function encryptRouteParameter($data)
    {
        $encryptedData = Crypt::encrypt($data);
        return $encryptedData;
    }
    public function decryptRouteParameter($encryptedData)
    {

        $decryptedData = Crypt::decrypt($encryptedData);
        return $decryptedData;

    }
    public function print_invoice($id, Request $request)
    {
        $input = $request->all();

        App::setLocale('ar');

        $this->data['lang']        = 'ar';
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-008' , $this->data['lang']);
        $this->data['master']      = PurchasesOrder::get_purchases_order_m($id, $this->data['lang']);
        $this->data['details']     = PurchasesOrder::get_purchases_order_d($id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);

        $temp_name                 = Setting::get_main_setting_by_code('purchases_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Purchases.purchases_order.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }
}
